<?php
/**
 * Create New Group
 *
 * @package     AutomatorWP\Integrations\BuddyPress\Actions\Create_New_Group
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyPress_Create_New_Group extends AutomatorWP_Integration_Action {

    public $integration = 'buddypress';
    public $action = 'buddypress_create_new_group';

    /**
     * The action user id
     *
     * @since 1.0.0
     *
     * @var int $user_id
     */
    public $user_id = 0;

    /**
     * The action post id
     *
     * @since 1.0.0
     *
     * @var int $post_id
     */
    public $post_id = 0;

    /**
     * The post meta
     *
     * @since 1.0.0
     *
     * @var array $post_meta
     */
    public $post_meta = array();

    /**
     * The action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Create a group', 'automatorwp-pro' ),
            'select_option'     => __( 'Create a <strong>group</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Group. */
            'edit_label'        => sprintf( __( 'Create a %1$s', 'automatorwp-pro' ), '{group}' ),
            /* translators: %1$s: Group. */
            'log_label'         => sprintf( __( 'Create a %1$s', 'automatorwp-pro' ), '{group}' ),
            'options'           => array(
                'group' => array(
                    'default' => __( 'group', 'automatorwp-pro' ),
                    'fields' => array(
                        'group_name' => array(
                            'name' => __( 'Group name:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'group_slug' => array(
                            'name' => __( 'Group slug:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'group_description' => array(
                            'name' => __( 'Group description:', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                        ),
                        'group_organizer' => array(
                            'name' => __( 'Group organizer ID:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID to set as group organizer. Leave blank to set the user who completes the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'add_user_as_member' => array(
                            'name' => __( 'Add user as group member:', 'automatorwp-pro' ),
                            'desc' => __( 'Check this option to add user who completed the automation as a member of this group. User only will get added as member is the user is not the group organizer.', 'automatorwp-pro' ),
                            'type' => 'checkbox',
                            'classes' => 'cmb2-switch',
                        ),
                        'group_status' => array(
                            'name' => __( 'Group privacy:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'public'    => __( 'Public', 'automatorwp-pro' ),
                                'private'   => __( 'Private', 'automatorwp-pro' ),
                                'hidden'    => __( 'Hidden', 'automatorwp-pro' ),
                            ),
                            'default' => 'public'
                        ),
                        'group_invite_status' => array(
                            'name' => __( 'Who can invite others to join this group?', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'members'   => __( 'All Members', 'automatorwp-pro' ),
                                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
                            ),
                            'default' => 'members'
                        ),
                        'group_feed_status' => array(
                            'name' => __( 'Who can post into this group?', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'members'   => __( 'All group members', 'automatorwp-pro' ),
                                'mods'      => __( 'Organizers and Moderators only', 'automatorwp-pro' ),
                                'admins'    => __( 'Organizers only', 'automatorwp-pro' ),
                            ),
                            'default' => 'members'
                        ),
                        'group_media_status' => array(
                            'name' => __( 'Who can manage photos in this group?', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'members'   => __( 'All Members', 'automatorwp-pro' ),
                                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
                            ),
                            'default' => 'members'
                        ),
                        'group_document_status' => array(
                            'name' => __( 'Which members of this group are allowed to manage documents?', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'members'   => __( 'All Members', 'automatorwp-pro' ),
                                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
                            ),
                            'default' => 'members'
                        ),
                        'group_album_status' => array(
                            'name' => __( 'Who can manage albums in this group?', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'members'   => __( 'All Members', 'automatorwp-pro' ),
                                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
                            ),
                            'default' => 'members'
                        ),
                        'group_message_status' => array(
                            'name' => __( 'Which members of this group are allowed to send group messages?', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'members'   => __( 'All Members', 'automatorwp-pro' ),
                                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
                            ),
                            'default' => 'members'
                        ),
                        'group_type' => automatorwp_utilities_post_field( array(
                            'name'                  => __( 'Group type:', 'automatorwp-pro' ),
                            'post_type'             => 'bp-group-type',
                            'placeholder'           => __( 'Select a group type', 'automatorwp-pro' ),
                            'option_none_label'     => __( 'Select a group type', 'automatorwp-pro' ),
                            'option_custom'         => true,
                            'option_custom_desc'    => __( 'Group Type ID', 'automatorwp-pro' ),
                        ) ),
                        'group_type_custom' => automatorwp_utilities_custom_field( array(
                            'option_custom_desc'    => __( 'Group Type ID', 'automatorwp-pro' ),
                        ) ),
                        'group_parent' => array(
                            'name' => __( 'Parent group:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-ajax-selector',
                            'option_none' => false,
                            'attributes' => array(
                                'data-action' => 'automatorwp_buddypress_get_groups',
                                'data-option-none' => false,
                                'data-placeholder' => __( 'Select a group', 'automatorwp-pro' ),
                            ),
                            'options_cb' => 'automatorwp_buddypress_options_cb_group',
                            'default' => ''
                        ),
                        'group_users' => array(
                            'name' => __( 'Add users to group:', 'automatorwp-pro' ),
                            'description' => __( 'The user ID to add them to the group.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'repeatable' => true
                        ),
                        'post_meta' => array(
                            'name' => __( 'Group Meta:', 'automatorwp-pro' ),
                            'desc' => __( 'The group meta values keyed by their meta key.', 'automatorwp-pro' ),
                            'type' => 'group',
                            'classes' => 'automatorwp-fields-table',
                            'options'     => array(
                                'add_button'        => __( 'Add meta', 'automatorwp-pro' ),
                                'remove_button'     => '<span class="dashicons dashicons-no-alt"></span>',
                            ),
                            'fields' => array(
                                'meta_key' => array(
                                    'name' => __( 'Meta Key:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                                'meta_value' => array(
                                    'name' => __( 'Meta Value:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                            ),
                        ),
                        'group_exists' => array(
                            'name' => __( 'What to do if the group already exist:', 'automatorwp-pro' ),
                            'desc' => __( 'Note: If you choose "Continue with the group creation", a new group will get created with a different slug.', 'automatorwp-pro' ),
                            'type' => 'radio',
                            'options' => array(
                                'continue' => __( 'Continue with the group creation', 'automatorwp-pro' ),
                                'add_user' => __( 'Add user to the existent group', 'automatorwp-pro' ),
                                'nothing' => __( 'Do nothing', 'automatorwp-pro' ),
                            ),
                            'default' => 'continue',
                        ),
                        'existing_group_field' => array(
                            'name' => __( 'Field that makes the group unique:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'name' => __( 'Name', 'automatorwp-pro' ),
                                'slug' => __( 'Slug', 'automatorwp-pro' ),
                            ),
                            'default' => 'name',
                        ),
                    ),
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        global $wpdb;

        // Bail if BuddyPress function does not exist
        if ( ! function_exists( 'groups_create_group' ) ) {
            $this->result = __( 'Groups component is not active.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $group_name = $action_options['group_name'];
        $group_slug = $action_options['group_slug'];
        $group_description = $action_options['group_description'];
        $group_organizer = $action_options['group_organizer'];
        $add_user_as_member = (bool) $action_options['add_user_as_member'];
        $group_status = $action_options['group_status'];
        $group_invite_status = $action_options['group_invite_status'];
        $group_feed_status = $action_options['group_feed_status'];
        $group_media_status = $action_options['group_media_status'];
        $group_document_status = $action_options['group_document_status'];
        $group_album_status = $action_options['group_album_status'];
        $group_message_status = $action_options['group_message_status'];
        $group_type = $action_options['group_type'];
        $group_parent = $action_options['group_parent'];
        $group_users = $action_options['group_users'];
        $group_exists = $action_options['group_exists'];
        $existing_group_field = $action_options['existing_group_field'];

        // Setup the group organizer
        $group_organizer = absint( $group_organizer );

        if( $group_organizer === 0 ) {
            $group_organizer = $user_id;
        }

        $this->user_id = $group_organizer;

        // Setup the group parent
        $group_parent = absint( $group_parent );

        if( $group_parent === 0 ) {
            $group_parent = null;
        }

        // Check if user wants to abort in case the group exists
        if( $group_exists !== 'continue' ) {
            $search_field = ( empty( $existing_group_field ) ? 'name' : $existing_group_field );
            $search_value = ( $search_field === 'slug' ? $group_slug : $group_name );

            // Get BuddyPress
            $bp = buddypress();

            // Check if group exists
            $existent_group_id = absint( $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$bp->groups->table_name} g WHERE g.{$search_field} = %s", $search_value ) ) );

            if( $existent_group_id !== 0 ) {

                switch ( $group_exists ) {
                    case 'add_user':
                        // Add user to the existent group
                        groups_join_group( $existent_group_id, $user_id );

                        $this->result = sprintf( __( 'Group not created because group found by the %1$s "%2$s", user has been added to the group', 'automatorwp-pro' ), $search_field, $search_value );
                        break;
                    case 'nothing':
                        // Do nothing
                        $this->result = sprintf( __( 'Group not created because group found by the %1$s "%2$s"', 'automatorwp-pro' ), $search_field, $search_value );
                        break;
                }

                // Prevent to coninue with the group creation
                return;

            }
        }

        $group_id = groups_create_group( array(
			'creator_id'   => $group_organizer,
			'name'         => $group_name,
			'slug'         => $group_slug,
			'description'  => $group_description,
			'status'       => $group_status,
            'parent_id'    => $group_parent,
        ) );

        // Bail if group could not get created
        if( ! $group_id ) {
            $this->result = __( 'Could not create the group', 'automatorwp-pro' );
            return;
        }

        $this->post_id = $group_id;
        $this->result = __( 'Group created successfully', 'automatorwp-pro' );

        // Set the group type
        $group_type = absint( $group_type );

        if( $group_type !== 0 ) {

            $group_type_post = get_post( $group_type );

            // If group type post exists, then update the group type
            if( $group_type_post ) {
                bp_groups_set_group_type( $group_id, $group_type_post->post_name, true );
            }
        }

        // Set the invite status
        groups_update_groupmeta( $group_id, 'invite_status', $group_invite_status );

        // Set the activity feed status
        groups_update_groupmeta( $group_id, 'activity_feed_status', $group_feed_status );

        // Set the media status
        groups_update_groupmeta( $group_id, 'media_status', $group_media_status );

        // Set the document status
        groups_update_groupmeta( $group_id, 'document_status', $group_document_status );

        // Set the album status
        groups_update_groupmeta( $group_id, 'album_status', $group_album_status );

        // Set the message status
        groups_update_groupmeta( $group_id, 'message_status', $group_message_status );

        if( $group_organizer !== $user_id && $add_user_as_member ) {
            groups_join_group( $group_id, $user_id );
        }

        // Add users to the group
        if ( ! empty ( $group_users ) ){
            foreach( $group_users as $user_id ){
                if ( get_userdata( $user_id ) ){
                    groups_join_group ( $group_id, $user_id );
                }
            }
        }

        if( is_array( $action_options['post_meta'] ) ) {

            foreach( $action_options['post_meta'] as $i => $meta ) {

                // Parse automation tags replacements to both, key and value
                $meta_key = automatorwp_parse_automation_tags( $automation->id, $user_id, $meta['meta_key'] );
                $meta_value = automatorwp_parse_automation_tags( $automation->id, $user_id, $meta['meta_value'] );

                // Sanitize
                $meta_key = sanitize_text_field( $meta_key );
                $meta_value = sanitize_text_field( $meta_value );

                // Update the group meta meta
                groups_update_groupmeta( $this->post_id, $meta_key, $meta_value );

                $this->post_meta[$meta_key] = $meta_value;

                // Update action options to be passed on upcoming hooks
                $action_options['post_meta'][$i] = array(
                    'meta_key' => $meta_key,
                    'meta_value' => $meta_value,
                );

            }

        }

        // Add filter to force the logged in user ID
        add_filter( 'bp_loggedin_user_id', array( $this, 'force_logged_in_user_id' ) );

        /**
         * Action to keep compatibility with "User creates a group" trigger.
         *
         * @since 1.0.0
         *
         * @param int $group_id ID of the newly created group.
         */
        do_action( 'groups_group_create_complete', $group_id );

        // Remove filter added previously
        remove_filter( 'bp_loggedin_user_id', array( $this, 'force_logged_in_user_id' ) );

    }

    /**
     * Force the logged in user ID for triggered hooks
     *
     * @param int $user_id
     *
     * @return int
     */
    public function force_logged_in_user_id( $user_id ) {
        return $this->user_id;
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log post ID
        add_filter( 'automatorwp_user_completed_action_post_id', array( $this, 'post_id' ), 10, 6 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Action custom log post ID
     *
     * @since 1.0.0
     *
     * @param int       $post_id            The post ID, by default 0
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return int
     */
    public function post_id( $post_id, $action, $user_id, $event, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $post_id;
        }

        if( $this->post_id ) {
            $post_id = $this->post_id;
        }

        return $post_id;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store post fields
        $group_fields = array(
            'group_name',
            'group_slug',
            'group_description',
            'group_organizer',
            'group_status',
            'group_invite_status',
            'group_feed_status',
            'group_media_status',
            'group_document_status',
            'group_album_status',
            'group_message_status',
            'group_type',
            'group_parent'
        );

        foreach( $group_fields as $group_field ) {
            $log_meta[$group_field] = $action_options[$group_field];
        }

        // Store post meta
        $log_meta['post_meta'] = $this->post_meta;

        // Store result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['group_info'] = array(
            'name' => __( 'Group Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about the group created.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['group_name'] = array(
            'name'      => __( 'Name:', 'automatorwp-pro' ),
            'desc'      => __( 'The group name.', 'automatorwp-pro' ),
            'type'      => 'text',
        );

        $log_fields['group_slug'] = array(
            'name'      => __( 'Slug:', 'automatorwp-pro' ),
            'desc'      => __( 'The group slug.', 'automatorwp-pro' ),
            'type'      => 'text',
        );

        $log_fields['group_description'] = array(
            'name'      => __( 'Description:', 'automatorwp-pro' ),
            'desc'      => __( 'The group description.', 'automatorwp-pro' ),
            'type'      => 'text',
            'wpautop'   => true,
        );

        $log_fields['group_status'] = array(
            'name' => __( 'Group privacy:', 'automatorwp-pro' ),
            'type' => 'select',
            'options' => array(
                'public'    => __( 'Public', 'automatorwp-pro' ),
                'private'   => __( 'Private', 'automatorwp-pro' ),
                'hidden'    => __( 'Hidden', 'automatorwp-pro' ),
            ),
        );

        $log_fields['group_invite_status'] = array(
            'name' => __( 'Who can invite others to join this group?', 'automatorwp-pro' ),
            'type' => 'select',
            'options' => array(
                'members'   => __( 'All Members', 'automatorwp-pro' ),
                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
            ),
        );

        $log_fields['group_feed_status'] = array(
            'name' => __( 'Who can post into this group?', 'automatorwp-pro' ),
            'type' => 'select',
            'options' => array(
                'members'   => __( 'All group members', 'automatorwp-pro' ),
                'mods'      => __( 'Organizers and Moderators only', 'automatorwp-pro' ),
                'admins'    => __( 'Organizers only', 'automatorwp-pro' ),
            ),
        );

        $log_fields['group_media_status'] = array(
            'name' => __( 'Who can manage photos in this group?', 'automatorwp-pro' ),
            'type' => 'select',
            'options' => array(
                'members'   => __( 'All Members', 'automatorwp-pro' ),
                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
            ),
        );

        $log_fields['group_document_status'] = array(
            'name' => __( 'Which members of this group are allowed to manage documents?', 'automatorwp-pro' ),
            'type' => 'select',
            'options' => array(
                'members'   => __( 'All Members', 'automatorwp-pro' ),
                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
            ),
        );

        $log_fields['group_album_status'] = array(
            'name' => __( 'Who can manage albums in this group?', 'automatorwp-pro' ),
            'type' => 'select',
            'options' => array(
                'members'   => __( 'All Members', 'automatorwp-pro' ),
                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
            ),
        );

        $log_fields['group_message_status'] = array(
            'name' => __( 'Which members of this group are allowed to send group messages?', 'automatorwp-pro' ),
            'type' => 'select',
            'options' => array(
                'members'   => __( 'All Members', 'automatorwp-pro' ),
                'mods'      => __( 'Organizers and Moderators', 'automatorwp-pro' ),
                'admins'    => __( 'Organizers', 'automatorwp-pro' ),
            ),
        );

        $log_fields['group_parent'] = array(
            'name'      => __( 'Parent ID:', 'automatorwp-pro' ),
            'desc'      => __( 'The group parent ID.', 'automatorwp-pro' ),
            'type'      => 'text',
        );

        $log_fields['post_meta'] = array(
            'name' => __( 'Group Meta:', 'automatorwp-pro' ),
            'desc' => __( 'The group meta values keyed by their meta key.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_BuddyPress_Create_New_Group();